/**
* \file: svg_font_demo.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: svg_font
*
* \copyright (c) 2010-2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include "EGL/egl.h"
#ifdef HAS_OPENGLES2
    #include "GLES2/gl2.h"
    #define HAS_OPENGLES 1
#endif

#ifdef HAS_OPENVG
    #include <VG/openvg.h>
    #include <VG/vgu.h>
#endif


#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <adit_typedef.h>
#include <svg_font.h>


#define SCREEN_WIDTH  800
#define SCREEN_HEIGHT 480

#ifdef USE_WAYLAND
	#include <wayland-client.h>
	#include <wayland-egl.h>
        #include <compositor-shim.h>
	#include "wayland-util.h"


typedef struct _WaylandGles WaylandGles;

struct _WaylandGles
{
  SVGUint32 surface_id;
  SVGUint32 layer_id;
  struct wl_display* wl_display;
  struct wl_surface* wl_surface;
  struct wl_registry* wl_registry;
  struct wl_egl_window* wl_native_window;
  struct wl_compositor* wl_compositor;

  struct compositor_shim_context* wlAdapterContext;
  struct compositor_shim_surface_context wlAdapterSurfaceContext;
};

static WaylandGles aWayland;

void RegistryHandleGlobal(void* data, struct wl_registry* registry, uint32_t name, const char* interface, uint32_t version)
{
  version = version;
  WaylandGles * sink = (WaylandGles *) data;

	if (!strcmp(interface, "wl_compositor")){
	    sink->wl_compositor =   (struct wl_compositor*)(wl_registry_bind(registry,
										  name,
										  &wl_compositor_interface,
										  1));
	}
}

static struct wl_registry_listener registryListener = {
    RegistryHandleGlobal,
    NULL
};

static SVGBoolean create_wl_context(WaylandGles * sink)
{
    sink->wl_display = wl_display_connect(NULL);
    if (sink->wl_display == NULL)
    {
  	    printf("fail wl 1\n");
        return FALSE;
    }

    sink->wl_registry = wl_display_get_registry(sink->wl_display);

    wl_registry_add_listener(sink->wl_registry, &registryListener, sink);

    wl_display_dispatch(sink->wl_display);
    wl_display_roundtrip(sink->wl_display);

    if(sink->wl_compositor == NULL)
    {
  	    printf("fail wl 2\n");
        return FALSE;
    }

    sink->wlAdapterContext = compositor_shim_initialize (sink->wl_display);
    if (NULL == sink->wlAdapterContext)
    {
  	    printf("fail wl 3\n");
        return FALSE;
    }

    sink->wl_surface = wl_compositor_create_surface(sink->wl_compositor);
    if(sink->wl_surface == NULL)
    {
  	    printf("fail wl 4\n");
        return FALSE;
    }

    sink->wl_native_window = wl_egl_window_create(sink->wl_surface, SCREEN_WIDTH, SCREEN_HEIGHT);
    if (sink->wl_native_window == NULL)
    {
  	    printf("fail wl 5\n");
        return FALSE;
    }

    return TRUE;
}
#endif /* USE_WAYLAND */

#ifdef HAS_OPENVG
static void vg_font_demo(void);
#endif
static void gl_font_demo(void);
static void draw_font_test(void);

static SVGInt32 runmode    = 0;
static SVGInt32 use_opengl = 0;

static SVGFontModule           Plugin_Freetype;

static SVGFont                 Font_AditChicago;
static SVGFontContext*         p_font_ctx;
static SVGFontContext*         p_font_ctx2;
static SVGUint8                a_string[] =  "is a 2-color teststring.";
static SVGUint8                a_string1[] = "This is an outline teststring.";
static SVGUint8                a_string2[] = "This is an extended teststring";
static SVGUint8                a_string3[] = "The vertical teststring";
static SVGFloat                a_angle = 0.0;
static SVGFloat                pos[62] = {0};
static SVGFloat                pos2[62] = {0};
static SVGFloat                angles[31] = {0};
static SVGUint8                thai_string[] = "123456ABCD thai_string \xe0\xb8\x99\xe0\xb9\x89\xe0\xb8\xb3\xe0\xb8\x9a\xe0\xb8\xa5\xe0\xb8\xb9\xe0\xb8\x97\xe0\xb8\xb9\xe0\xb8\x98";
//static SVGUint8              arabic_string[] = "123456ABCD gnirts_cibara \xd9\x86\xd8\xaa\xd8\xa8\xd8\xa4\xd8\xa9\xd8\xb2\xd8\xb8\xd8\xa6\xd8\xb4\xd8\xab\xd8\xba\xd9\x86";

static EGLDisplay       eglDisplay=EGL_NO_DISPLAY;
static EGLConfig        eglConfig;
static EGLContext       eglContext=EGL_NO_CONTEXT;
static EGLSurface       eglWindowSurface=EGL_NO_SURFACE;

#ifdef HAS_OPENVG
static VGPaint 				   testpaint;
static VGPath                  a_path;
#endif

#ifdef HAS_OPENGLES2
GLfloat projection[4][4] = {{2.0/800.0,0,0,0},{0,2.0/480.0,0,0},{0,0,0,0},{-1.0,-1.0,0,1}};

static char const * const apszFiles[4]=
{   /* shader for drawing lines and boxes */
    "attribute vec4 position;                                     \
     uniform mat4 mvp;                                            \
     void main(void)                                              \
     {                                                            \
        gl_Position = mvp * position;                             \
     }",                                                         
                                                                 
    "uniform mediump vec4 basecolor;                              \
     void main(void)                                              \
     {                                                            \
        gl_FragColor = basecolor;                                 \
     }",                                                          
    
    /* shader for drawing glyps */                                                             
    "varying vec2 texcoord;                                       \
     attribute vec4 position;                                     \
     attribute vec2 inputtexcoord;                                \
     uniform mat4 mvp;                                            \
     void main(void)                                              \
     {                                                            \
        texcoord = inputtexcoord;                                 \
        gl_Position = mvp * position;                             \
     }",                                                         
                                                                 
    "varying highp vec2 texcoord;                                 \
     uniform int noantialias;                                \
     uniform mediump float opacity;                               \
     uniform mediump vec4 basecolor;                              \
     uniform sampler2D basetexture;                               \
     void main(void)                                              \
     {                           \
                                   \
        mediump vec4 texlookup = texture2D(basetexture, texcoord);\
        gl_FragColor = basecolor;                                 \
        if(noantialias == 0)                                      \
          gl_FragColor.a = opacity * texlookup.a;                 \
        else                                                      \
         if (0.0 == texlookup.a)                                  \
           gl_FragColor.a = 0.0;                                  \
         else                                                     \
         gl_FragColor.a = opacity;                                \
     }"                                     
};                                          
                                            
SVGInt32 hProgramHandle[2];                         
SVGInt32 colorlocation;
SVGInt32 colorlocation1;
SVGInt32 noaalocation;
SVGInt32 opacitylocation;
SVGInt32 mvplocation;
SVGInt32 mvplocation1;
SVGInt32 texturelocation;
SVGInt32 vertexlocation;
SVGFloat modelview[4][4];
SVGFloat mvp[4][4];

void demo_matrixMutiply(SVGFloat psRes[4][4], SVGFloat psSrcA[4][4], SVGFloat psSrcB[4][4])
{
	SVGFloat fB00, fB01, fB02, fB03;
	SVGFloat fB10, fB11, fB12, fB13;
	SVGFloat fB20, fB21, fB22, fB23;
	SVGFloat fB30, fB31, fB32, fB33;
	SVGUint32 i;

	fB00 = psSrcB[0][0]; fB01 = psSrcB[0][1]; fB02 = psSrcB[0][2]; fB03 = psSrcB[0][3];
	fB10 = psSrcB[1][0]; fB11 = psSrcB[1][1]; fB12 = psSrcB[1][2]; fB13 = psSrcB[1][3];
	fB20 = psSrcB[2][0]; fB21 = psSrcB[2][1]; fB22 = psSrcB[2][2]; fB23 = psSrcB[2][3];
	fB30 = psSrcB[3][0]; fB31 = psSrcB[3][1]; fB32 = psSrcB[3][2]; fB33 = psSrcB[3][3];

	for (i = 0; i < 4; i++) 
	{
		psRes[i][0] = psSrcA[i][0]*fB00 + psSrcA[i][1]*fB10	+ psSrcA[i][2]*fB20 + psSrcA[i][3]*fB30;
		psRes[i][1] = psSrcA[i][0]*fB01 + psSrcA[i][1]*fB11	+ psSrcA[i][2]*fB21 + psSrcA[i][3]*fB31;
		psRes[i][2] = psSrcA[i][0]*fB02 + psSrcA[i][1]*fB12	+ psSrcA[i][2]*fB22 + psSrcA[i][3]*fB32;
		psRes[i][3] = psSrcA[i][0]*fB03 + psSrcA[i][1]*fB13	+ psSrcA[i][2]*fB23 + psSrcA[i][3]*fB33;
	}
}

void demo_matrixTranslate(SVGFloat pMatrix[4][4], SVGFloat fX, SVGFloat fY)
{
	pMatrix[3][0] += fX * pMatrix[0][0] + fY * pMatrix[1][0];
	pMatrix[3][1] += fX * pMatrix[0][1] + fY * pMatrix[1][1];
	pMatrix[3][2] += fX * pMatrix[0][2] + fY * pMatrix[1][2];
	pMatrix[3][3] += fX * pMatrix[0][3] + fY * pMatrix[1][3];
}

void demo_matrixIdentity(SVGFloat pMatrix[4][4])
{
	pMatrix[0][0] = 1.0f;
	pMatrix[0][1] = 0.0f;
	pMatrix[0][2] = 0.0f;
	pMatrix[0][3] = 0.0f;
	pMatrix[1][0] = 0.0f;
	pMatrix[1][1] = 1.0f;
	pMatrix[1][2] = 0.0f;
	pMatrix[1][3] = 0.0f;
	pMatrix[2][0] = 0.0f;
	pMatrix[2][1] = 0.0f;
	pMatrix[2][2] = 1.0f;
	pMatrix[2][3] = 0.0f;
	pMatrix[3][0] = 0.0f;
	pMatrix[3][1] = 0.0f;
	pMatrix[3][2] = 0.0f;
	pMatrix[3][3] = 1.0f;
}

static void handle_openGL_error(int line)
{
    unsigned int error_api = glGetError();
    if(GL_NO_ERROR != error_api)
    {
   	  fprintf(stderr," OpenGl error %d after line %d\n", error_api, line);
    }

}

void init(void)
{

    GLuint i;
    char pszInfoLog[1024];
    SVGInt32  nShaderStatus, nInfoLogLength;
    SVGInt32 hShaderHandle[4];

    /* Create a program object */
    hShaderHandle[0] = glCreateShader(GL_VERTEX_SHADER);
    handle_openGL_error(__LINE__);
    hShaderHandle[1] = glCreateShader(GL_FRAGMENT_SHADER);
    handle_openGL_error(__LINE__);
    hProgramHandle[0] = glCreateProgram();

    hShaderHandle[2] = glCreateShader(GL_VERTEX_SHADER);
    handle_openGL_error(__LINE__);
    hShaderHandle[3] = glCreateShader(GL_FRAGMENT_SHADER);
    handle_openGL_error(__LINE__);
    hProgramHandle[1] = glCreateProgram();
    handle_openGL_error(__LINE__);

    SVGInt32  nProgramLength;
    for(i=0; i<4 ; i++)
    {
        nProgramLength = strlen(apszFiles[i]);
        glShaderSource(hShaderHandle[i], 1, (const char **)&apszFiles[i], &nProgramLength);
        /* Compile the shader */
        glCompileShader(hShaderHandle[i]);
        handle_openGL_error(__LINE__);
        /* Check it compiled OK */
        glGetShaderiv(hShaderHandle[i], GL_COMPILE_STATUS, &nShaderStatus);
        if (nShaderStatus != GL_TRUE)
        {
            printf("Error: Failed to compile GLSL shader #%d\n",i);
            glGetShaderInfoLog(hShaderHandle[i], 1024, &nInfoLogLength, pszInfoLog);
            printf("%s",pszInfoLog);
            return ;
        }
    }
    /* Attach the shader to the programs */
    glAttachShader(hProgramHandle[0], hShaderHandle[0]);
    glAttachShader(hProgramHandle[0], hShaderHandle[1]);

    glBindAttribLocation(hProgramHandle[0], 0, "position");

    /* Link the program */
    glLinkProgram(hProgramHandle[0]);

    /* Check it linked OK */
    glGetProgramiv(hProgramHandle[0], GL_LINK_STATUS, &nShaderStatus);

    if (nShaderStatus != GL_TRUE)
    {
        printf("Error: Failed to link GLSL program\n");
        glGetProgramInfoLog(hProgramHandle[0], 1024, &nInfoLogLength, pszInfoLog);
        printf("%s",pszInfoLog);
        return ;
    }

    glValidateProgram(hProgramHandle[0]);

    glGetProgramiv(hProgramHandle[0], GL_VALIDATE_STATUS, &nShaderStatus);

    if (nShaderStatus != GL_TRUE)
    {
        printf("Error: Failed to validate GLSL program\n");
        glGetProgramInfoLog(hProgramHandle[0], 1024, &nInfoLogLength, pszInfoLog);
        printf("%s",pszInfoLog);
        return;
    }

    /* begin of glyph shader initialisation */
    glAttachShader(hProgramHandle[1], hShaderHandle[2]);
    glAttachShader(hProgramHandle[1], hShaderHandle[3]);

    glBindAttribLocation(hProgramHandle[1], 0, "position");

    /* Link the program */
    glLinkProgram(hProgramHandle[1]);

    /* Check it linked OK */
    glGetProgramiv(hProgramHandle[1], GL_LINK_STATUS, &nShaderStatus);

    if (nShaderStatus != GL_TRUE)
    {
        printf("Error: Failed to link GLSL program\n");
        glGetProgramInfoLog(hProgramHandle[1], 1024, &nInfoLogLength, pszInfoLog);
        printf("%s",pszInfoLog);
        return ;
    }

    glValidateProgram(hProgramHandle[1]);

    glGetProgramiv(hProgramHandle[1], GL_VALIDATE_STATUS, &nShaderStatus);

    if (nShaderStatus != GL_TRUE)
    {
        printf("Error: Failed to validate GLSL program\n");
        glGetProgramInfoLog(hProgramHandle[1], 1024, &nInfoLogLength, pszInfoLog);
        printf("%s",pszInfoLog);
        return;
    }

    glUseProgram(hProgramHandle[1]);
    glUniform1i(glGetUniformLocation(hProgramHandle[1], "basetexture"), 0);

    mvplocation1    = glGetUniformLocation(hProgramHandle[1], "mvp");
    texturelocation = glGetAttribLocation(hProgramHandle[1], "inputtexcoord");
    colorlocation1  = glGetUniformLocation(hProgramHandle[1], "basecolor");
    opacitylocation = glGetUniformLocation(hProgramHandle[1], "opacity");
    noaalocation    = glGetUniformLocation(hProgramHandle[1], "noantialias");
    vertexlocation  = glGetAttribLocation(hProgramHandle[1], "position");


    /* end of glyph shader initialisation */

    glUseProgram(hProgramHandle[0]);

    mvplocation = glGetUniformLocation(hProgramHandle[0], "mvp");
    glUniformMatrix4fv(mvplocation, 1, GL_FALSE, &projection[0][0]);
    glEnableVertexAttribArray (0);

    colorlocation = glGetUniformLocation(hProgramHandle[0], "basecolor");

    return;
}
#endif /* HAS_OPENGLES2 */

/*main function for a process or system program*/
#if defined(SYSTEM_PROGRAM) || defined (USER_PROCESS)
SVGInt32 main(int argc, char** argv)
{
	SVGInt32 rc = 0;

	if((0 == runmode) && (2 == argc))
	{
		runmode = 100000000;
		if( !strcmp(argv[1],"vg") )
		{
			use_opengl = 1;
			printf("OpenVG not longer supported! Font demo using OpenGL-ES instead\n");
			gl_font_demo();
		}
		else if( !strcmp(argv[1],"gles") )
		{
			use_opengl = 1;
			printf("Font demo using OpenGL-ES\n");
			gl_font_demo();
		}
	}
	else if((0 == runmode) && (3 == argc))
	{
		if( !strcmp(argv[1],"vg") )
		{
			use_opengl = 1;
			printf("OpenVG not longer supported! Font demo using OpenGL-ES instead\n");
			runmode = atoi(argv[2]);
			gl_font_demo();
		}
		else if( !strcmp(argv[1],"gles") )
		{
			use_opengl = 1;
			printf("Font demo using OpenGL-ES\n");
			runmode = atoi(argv[2]);
			gl_font_demo();
		}
	}
	else
	{
		printf("default Font demo using OpenGL-ES\n");
		runmode = 10000000;
		use_opengl = 1;
		gl_font_demo();
	}

	return rc;
}
#endif



static void handle_egl_error(const char *name)
{
  static char * const error_strings[] =
    {
      "EGL_SUCCESS",
      "EGL_NOT_INITIALIZED",
      "EGL_BAD_ACCESS",
      "EGL_BAD_ALLOC",
      "EGL_BAD_ATTRIBUTE",
      "EGL_BAD_CONFIG",
      "EGL_BAD_CONTEXT",
      "EGL_BAD_CURRENT_SURFACE",
      "EGL_BAD_DISPLAY",
      "EGL_BAD_MATCH",
      "EGL_BAD_NATIVE_PIXMAP",
      "EGL_BAD_NATIVE_WINDOW",
      "EGL_BAD_PARAMETER",
      "EGL_BAD_SURFACE"
    };
  EGLint error_code=eglGetError();

  fprintf(stderr," %s: egl error \"%s\" (0x%x)\n", name,
          error_strings[error_code-EGL_SUCCESS], error_code);
  exit(EXIT_FAILURE);
}

static void eglClose(void)
{
	eglMakeCurrent(eglDisplay, NULL, NULL, NULL);
    eglDestroyContext(eglDisplay, eglContext);
    eglDestroySurface(eglDisplay, eglWindowSurface);
    eglTerminate(eglDisplay);
    eglReleaseThread();
#ifdef USE_WAYLAND
	  WaylandGles* sink = &aWayland;
	  wl_display_flush( sink->wl_display );
	  wl_display_roundtrip( sink->wl_display);
          compositor_shim_terminate(sink->wlAdapterContext);

	  if (sink->wl_native_window)
	  {
	      wl_egl_window_destroy(sink->wl_native_window);
	  }

	  if (sink->wl_surface)
	  {
	      wl_surface_destroy(sink->wl_surface);
	  }

	  if (sink->wl_compositor)
	  {
	      wl_compositor_destroy(sink->wl_compositor);
	  }
	  if (sink->wl_display)
	  {
		  wl_display_disconnect( sink->wl_display);
	  }
#endif /* USE_WAYLAND */
}

static void eglOpen(int api)
{
  EGLBoolean ret;
  static const EGLint gl_context_attribs[] =
    {
      EGL_CONTEXT_CLIENT_VERSION, 2,
      EGL_NONE
    };

   EGLint s_configAttribs[] =
    {
      EGL_RED_SIZE,	5,
      EGL_GREEN_SIZE, 	6,
      EGL_BLUE_SIZE,	5,
      EGL_ALPHA_SIZE, 	EGL_DONT_CARE,
      EGL_DEPTH_SIZE,         EGL_DONT_CARE,
      EGL_STENCIL_SIZE,       EGL_DONT_CARE,
      EGL_SURFACE_TYPE,           EGL_WINDOW_BIT,
      EGL_SAMPLES,            EGL_DONT_CARE,
      EGL_RENDERABLE_TYPE,    EGL_OPENVG_BIT,
      EGL_NONE
    };
  EGLint numConfigs;
  EGLint majorVersion;
  EGLint minorVersion;

#ifndef USE_WAYLAND
  EGLNativeWindowType  native_window  = fbCreateWindow(fbGetDisplay(NULL), 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT);
#else
  /* set wayland environment to run without shell script */
  if(NULL == getenv("XDG_RUNTIME_DIR"))
  {
	  setenv("XDG_RUNTIME_DIR","/tmp/", TRUE);
  }

  WaylandGles* sink = &aWayland;
  int adapter_error = 0;
  sink->surface_id = 40;
  sink->layer_id = 3000;
  if (!create_wl_context(sink))
  {
	  printf("fail 2\n");
//	  eglClose();
	  return ;
  }

  eglDisplay = eglGetDisplay( (EGLNativeDisplayType)sink->wl_display);

  adapter_error = compositor_shim_surface_init (&sink->wlAdapterSurfaceContext, sink->wl_surface, sink->layer_id, 
                                                sink->surface_id, SCREEN_WIDTH, SCREEN_HEIGHT);
  if (adapter_error != 0)
  {
      printf("compositor_shim_surface_init() failed\n");
      return;
  }

  adapter_error = compositor_shim_surface_configure (sink->wlAdapterContext, &sink->wlAdapterSurfaceContext, ADAPTER_CONFIGURATION_ALL);
  if (adapter_error != 0)
  {
      printf("compositor_shim_surface_configure() failed\n");
      return;
  }
#endif /* USE_WAYLAND */

  if(api > 0)
  {
	  s_configAttribs[16] =  EGL_NONE;
    eglBindAPI(EGL_OPENGL_ES_API);
  }
  else
  {
    eglBindAPI(EGL_OPENVG_API);
  }
#ifndef USE_WAYLAND
  eglDisplay = eglGetDisplay(EGL_DEFAULT_DISPLAY);
#endif

  eglInitialize(eglDisplay, &majorVersion, &minorVersion);

  eglGetConfigs(eglDisplay, NULL, 0, &numConfigs);

  eglChooseConfig(eglDisplay, s_configAttribs, &eglConfig, 1, &numConfigs);
#ifndef USE_WAYLAND
  eglWindowSurface = eglCreateWindowSurface(eglDisplay, eglConfig, native_window, NULL);
#else
  eglWindowSurface = eglCreateWindowSurface(eglDisplay, eglConfig, (EGLNativeWindowType)sink->wl_native_window, NULL);
#endif
  if (eglWindowSurface == EGL_NO_SURFACE)
    {
      handle_egl_error("eglCreateWindowSurface");
    }

  if(api > 0)
  {
	  eglContext = eglCreateContext(eglDisplay, eglConfig, EGL_NO_CONTEXT, gl_context_attribs);
  }
  else
  {
	  eglContext = eglCreateContext(eglDisplay, eglConfig, EGL_NO_CONTEXT, NULL);
  }

  if (eglContext == EGL_NO_CONTEXT)
    {
      handle_egl_error("eglCreateContext");
    }

  ret = eglMakeCurrent(eglDisplay, eglWindowSurface, eglWindowSurface, eglContext);
  if (ret == EGL_FALSE)
    handle_egl_error("eglMakeCurrent");
  else
    printf("setup ok\n" );
  eglSwapInterval(eglDisplay,1);

}

#ifdef HAS_OPENVG
void vg_font_demo(void)
{
	VGfloat clearcolor[4] = { 0.0f, 0.0f, 1.0f, 1.0f };

	eglOpen(0);
	
    svgInitFonts();
    svgGetAvailableFontModules(&Plugin_Freetype, 0, SVG_TOP);
    while (Plugin_Freetype.id != SVG_PLUGIN_FREETYPE)
    {
        svgGetAvailableFontModules(&Plugin_Freetype, 0, SVG_NEXT);
    }

    p_font_ctx = svgCreateFontContext(&Plugin_Freetype, &eglContext, NULL);
    
    svgGetAvailableFonts( p_font_ctx, &Font_AditChicago, 0, 0, SVG_TOP );
    svgLoadFont( p_font_ctx, &Font_AditChicago, SVG_ENCODING_ASCII, 40);

    /*Initialize framework:*/
    testpaint = vgCreatePaint ();
	vgSetPaint (testpaint, VG_FILL_PATH | VG_STROKE_PATH);
    /* set fill color to red, so 2-color sting filled area is red */
	vgSetColor (testpaint,  0xFF0000FF);
    vgSeti (VG_MASKING,    VG_FALSE);
    vgSeti (VG_SCISSORING, VG_FALSE);
    vgSeti (VG_IMAGE_MODE, /*VG_DRAW_IMAGE_STENCIL*/VG_DRAW_IMAGE_MULTIPLY);
    vgSeti (VG_MATRIX_MODE, VG_MATRIX_FILL_PAINT_TO_USER);    vgLoadIdentity();
    vgSeti (VG_MATRIX_MODE, VG_MATRIX_STROKE_PAINT_TO_USER);   vgLoadIdentity();
    vgSeti (VG_MATRIX_MODE, VG_MATRIX_IMAGE_USER_TO_SURFACE);  vgLoadIdentity();
    vgSeti (VG_MATRIX_MODE, VG_MATRIX_PATH_USER_TO_SURFACE);   vgLoadIdentity();
    vgSeti (VG_BLEND_MODE, VG_BLEND_SRC_OVER);
    vgSetfv (VG_STROKE_DASH_PATTERN, 0, 0);
    a_path = vgCreatePath (VG_PATH_FORMAT_STANDARD, VG_PATH_DATATYPE_F, 1.0f, 0.0f, 0,0,VG_PATH_CAPABILITY_ALL);
    vgSeti(VG_RENDERING_QUALITY,  VG_RENDERING_QUALITY_FASTER);
    
    while ( runmode )
    {
    	vgSetfv (VG_CLEAR_COLOR, 4, clearcolor);
    	vgClear (0,0,SCREEN_WIDTH,SCREEN_HEIGHT);    
    	vgSeti(VG_STROKE_LINE_WIDTH, 2);
        draw_font_test();
        eglSwapBuffers(eglDisplay, eglWindowSurface);
#ifdef USE_WAYLAND

        WaylandGles* sink = &aWayland;
        struct wl_callback* callback = wl_surface_frame(sink->wl_surface);
        wl_callback_add_listener(callback, &framelistener, NULL);
        wl_display_flush(sink->wl_display);

#endif
		runmode--;  
    }
    vgDestroyPaint (testpaint);
    svgDestroyFontContext(p_font_ctx);
    svgCloseFonts();

    eglClose();
}
#endif    

void gl_font_demo(void)
{
#ifdef HAS_OPENGLES

    eglOpen(1);

    svgInitFonts();
    svgGetAvailableFontModules(&Plugin_Freetype, 0, SVG_TOP);
    while (Plugin_Freetype.id != SVG_PLUGIN_FREETYPE)
    {
        svgGetAvailableFontModules(&Plugin_Freetype, 0, SVG_NEXT);
    }

    p_font_ctx = svgCreateFontContext(&Plugin_Freetype, &eglContext, NULL);
    
    svgGetAvailableFonts( p_font_ctx, &Font_AditChicago, 0, 0, SVG_TOP );
    svgLoadFont( p_font_ctx, &Font_AditChicago, SVG_ENCODING_ASCII, 40);

    svgSetBodyColor(p_font_ctx, 0xff0000ff);        

    /* just for testing new font features */
    p_font_ctx2 = svgCreateFontContext(&Plugin_Freetype, &eglContext, NULL);

    svgGetAvailableFonts( p_font_ctx2, &Font_AditChicago, 0, 0, SVG_TOP );
    svgLoadFont( p_font_ctx2, &Font_AditChicago, SVG_ENCODING_ASCII, 270);

    svgSetBodyColor(p_font_ctx2, 0xff0000ff);
    svgSetCacheableGlyphCount(p_font_ctx2,17,NULL);

    svgSetCacheableGlyphCount(p_font_ctx2,25,NULL);

    svgSetCacheableGlyphCount(p_font_ctx2,36,NULL);

    svgSetCacheableGlyphCount(p_font_ctx2,49,NULL);


    svgSetCacheableGlyphCount(p_font_ctx2,150,NULL);

    svgSetCacheableGlyphCount(p_font_ctx2,200,NULL);

    svgSetCacheableGlyphCount(p_font_ctx2,2000,NULL);

    svgSetCacheableGlyphCount(p_font_ctx2,1000,NULL);

    svgSetCacheableGlyphCount(p_font_ctx2,1100,NULL);
    svgSetCacheableGlyphCount(p_font_ctx2,1200,NULL);
    svgSetCacheableGlyphCount(p_font_ctx2,1300,NULL);
    svgSetCacheableGlyphCount(p_font_ctx2,1400,NULL);
    svgSetCacheableGlyphCount(p_font_ctx2,1500,NULL);
    svgSetCacheableGlyphCount(p_font_ctx2,2001,NULL);

    init();

    while ( runmode )
    {
    	glClearColor(0.0f, 0.0f, 1.0f, 1.0f);				
        glClear(GL_COLOR_BUFFER_BIT);
        svgSetContourWidth(p_font_ctx,1);
        draw_font_test();
        eglSwapBuffers(eglDisplay, eglWindowSurface);
#ifdef USE_WAYLAND
        /*
        WaylandGles* sink = &aWayland;
        struct wl_callback* callback = wl_surface_frame(sink->wl_surface);
        wl_callback_add_listener(callback, &framelistener, NULL);
        wl_display_flush(sink->wl_display);
        */
#endif
		runmode--;
    }

    /* just for testing new open type font features */
    p_font_ctx2 = svgCreateFontContext(&Plugin_Freetype, &eglContext, NULL);

    svgGetAvailableFonts( p_font_ctx2, &Font_AditChicago, (SVGUint8*)"NotoSans-Regular", 0, SVG_TOP );
    printf("Selected font: %s %s\n", Font_AditChicago.fontName ,Font_AditChicago.filename);
    printf("Font demo using OpenGLES for open Type(thai) fonts\n");
    svgLoadFont( p_font_ctx2, &Font_AditChicago, SVG_ENCODING_UTF8, 40);

    svgSetBodyColor(p_font_ctx2, 0xff0000ff);
    glClearColor(0.0f, 1.0f, 1.0f, 1.0f);				
    glClear(GL_COLOR_BUFFER_BIT);
    svgSetContourWidth(p_font_ctx2,1);

    svgDrawFontString(p_font_ctx2, 50.0 ,30.0 , &thai_string[0], strlen((char*)thai_string));

    svgDrawFontString(p_font_ctx2, 300.0 ,70.0 , (SVGUint8*)"\xe0\xb8\xa1\xe0\xb8\x99\xe0\xb8\xb5\xe0\xb9\x89 "
                                                            "\xe0\xb8\x81\xe0\xb8\xb2\xe0\xb8\xa3\xe0\xb8\x95"
                                                            "\xe0\xb8\xb1\xe0\xb9\x89\xe0\xb8\x87\xe0\xb8\x84"
                                                            "\xe0\xb9\x88\xe0\xb8\xb2\xe0\xb9\x83\xe0\xb8\x81"
                                                            "\xe0\xb8\xa5\xe0\xb9\x89\xe0\xb8\x9b\xe0\xb8\xb1"
                                                            "\xe0\xb9\x8a\xe0\xb8\xa1\xe0\xb8\x99\xe0\xb9\x89"
                                                            "\xe0\xb8\xb3\xe0\xb8\xa1\xe0\xb8\xb1\xe0\xb8\x99",
            sizeof("\xe0\xb8\xa1\xe0\xb8\x99\xe0\xb8\xb5\xe0\xb9\x89"
                   "\xe0\xb8\x81\xe0\xb8\xb2\xe0\xb8\xa3\xe0\xb8\x95"
                   "\xe0\xb8\xb1\xe0\xb9\x89\xe0\xb8\x87\xe0\xb8\x84"
                   "\xe0\xb9\x88\xe0\xb8\xb2\xe0\xb9\x83\xe0\xb8\x81"
                   "\xe0\xb8\xa5\xe0\xb9\x89\xe0\xb8\x9b\xe0\xb8\xb1"
                   "\xe0\xb9\x8a\xe0\xb8\xa1\xe0\xb8\x99\xe0\xb9\x89"
                   "\xe0\xb8\xb3\xe0\xb8\xa1\xe0\xb8\xb1\xe0\xb8\x99"));
    svgDrawFontString(p_font_ctx2, 50.0 ,70.0 , (SVGUint8*)"\xe0\xb8\x95\xe0\xb8\xb1\xe0\xb9\x89\xe0\xb8\x87"
                                                           "\xe0\xb8\x84\xe0\xb9\x88\xe0\xb8\xb2\xe0\xb9\x82"
                                                           "\xe0\xb8\x97\xe0\xb8\xa3\xe0\xb8\xa8\xe0\xb8\xb1"
                                                           "\xe0\xb8\x9e\xe0\xb8\x97\xe0\xb9\x8c",
            sizeof("\xe0\xb8\x95\xe0\xb8\xb1\xe0\xb9\x89\xe0\xb8\x87"
                  "\xe0\xb8\x84\xe0\xb9\x88\xe0\xb8\xb2\xe0\xb9\x82"
                  "\xe0\xb8\x97\xe0\xb8\xa3\xe0\xb8\xa8\xe0\xb8\xb1"
                  "\xe0\xb8\x9e\xe0\xb8\x97\xe0\xb9\x8c"));
    svgDrawFontString(p_font_ctx2, 50.0 ,140.0 , (SVGUint8*)"\xe0\xb8\x95\xe0\xb8\xb1\xe0\xb9\x89\xe0\xb8\x87"
                                                            "\xe0\xb8\x84\xe0\xb9\x88\xe0\xb8\xb2 "
                                                            "\xe0\xb8\x9a\xe0\xb8\xa5\xe0\xb8\xb9\xe0\xb8\x97"
                                                            "\xe0\xb8\xb9\xe0\xb8\x98",
            sizeof("\xe0\xb8\x95\xe0\xb8\xb1\xe0\xb9\x89\xe0\xb8\x87"
                   "\xe0\xb8\x84\xe0\xb9\x88\xe0\xb8\xb2 \xe0\xb8\x9a"
                   "\xe0\xb8\xa5\xe0\xb8\xb9\xe0\xb8\x97\xe0\xb8\xb9"
                   "\xe0\xb8\x98"));
    svgDrawFontString(p_font_ctx2, 50.0 ,210.0 , (SVGUint8*)"\xe0\xb9\x80\xe0\xb8\x94\xe0\xb8\xb5\xe0\xb9\x8b"
                                                            "\xe0\xb8\xa2\xe0\xb8\xa7\xe0\xb8\x99\xe0\xb8\xb5"
                                                            "\xe0\xb9\x89",
            sizeof("\xe0\xb9\x80\xe0\xb8\x94\xe0\xb8\xb5\xe0\xb9\x8b"
                   "\xe0\xb8\xa2\xe0\xb8\xa7\xe0\xb8\x99\xe0\xb8\xb5"
                   "\xe0\xb9\x89"));
    svgDrawFontString(p_font_ctx2, 50.0 ,280.0 , (SVGUint8*)"\xe0\xb8\x9b\xe0\xb8\xb1\xe0\xb9\x8a\xe0\xb8\xa1"
                                                            "\xe0\xb8\x99\xe0\xb9\x89\xe0\xb8\xb3\xe0\xb8\xa1"
                                                            "\xe0\xb8\xb1\xe0\xb8\x99\n\xe0\xb9\x83\xe0"
                                                            "\xb8\x81\xe0\xb8\xa5\xe0\xb9\x89\xe0\xb9\x80\xe0"
                                                            "\xb8\x84\xe0\xb8\xb5\xe0\xb8\xa2\xe0\xb8\x87",
            sizeof("\xe0\xb8\x9b\xe0\xb8\xb1\xe0\xb9\x8a\xe0\xb8\xa1"
                   "\xe0\xb8\x99\xe0\xb9\x89\xe0\xb8\xb3\xe0\xb8\xa1"
                   "\xe0\xb8\xb1\xe0\xb8\x99\n\xe0\xb9\x83\xe0"
                   "\xb8\x81\xe0\xb8\xa5\xe0\xb9\x89\xe0\xb9\x80\xe0"
                   "\xb8\x84\xe0\xb8\xb5\xe0\xb8\xa2\xe0\xb8\x87"));
    svgDrawFontString(p_font_ctx2, 50.0 ,350.0 , (SVGUint8*)"\xe0\xb9\x83\xe0\xb8\x99\xe0\xb9\x80\xe0\xb8\xa1"
                                                            "\xe0\xb8\xb7\xe0\xb8\xad\xe0\xb8\x87\xe0\xb8\xad"
                                                            "\xe0\xb8\xb7\xe0\xb9\x88\xe0\xb8\x99 ",
            sizeof("\xe0\xb9\x83\xe0\xb8\x99\xe0\xb9\x80\xe0\xb8\xa1"
                   "\xe0\xb8\xb7\xe0\xb8\xad\xe0\xb8\x87\xe0\xb8\xad"
                   "\xe0\xb8\xb7\xe0\xb9\x88\xe0\xb8\x99 "));
    svgDrawFontString(p_font_ctx2, 50.0 ,420.0 , (SVGUint8*)"\xe0\xb8\x88\xe0\xb8\xb8\xe0\xb8\x94\xe0\xb8\x97"
                                                            "\xe0\xb8\xb5\xe0\xb9\x88\xe0\xb8\xaa\xe0\xb8\x99"
                                                            "\xe0\xb9\x83\xe0\xb8\x88 \xe0\xb9\x83\xe0\xb8\x81"
                                                            "\xe0\xb8\xa5\xe0\xb9\x89\xe0\xb9\x80\xe0\xb8\x84"
                                                            "\xe0\xb8\xb5\xe0\xb8\xa2\xe0\xb8\x87",
            sizeof("\xe0\xb8\x88\xe0\xb8\xb8\xe0\xb8\x94\xe0\xb8\x97"
                   "\xe0\xb8\xb5\xe0\xb9\x88\xe0\xb8\xaa\xe0\xb8\x99"
                   "\xe0\xb9\x83\xe0\xb8\x88 \xe0\xb9\x83\xe0\xb8\x81"
                   "\xe0\xb8\xa5\xe0\xb9\x89\xe0\xb9\x80\xe0\xb8\x84"
                   "\xe0\xb8\xb5\xe0\xb8\xa2\xe0\xb8\x87"));
    svgDrawFontString(p_font_ctx2, 50.0 ,450.0 , (SVGUint8*)"\xe0\xb8\x94\xe0\xb8\xb5\xe0\xb9\x8b",
            sizeof("\xe0\xb8\x94\xe0\xb8\xb5\xe0\xb9\x8b"));

    eglSwapBuffers(eglDisplay, eglWindowSurface);
    sleep(10);

/*
//    Load font with arabic support FOR DISPLAYING Arabic characters
//    svgGetAvailableFonts( p_font_ctx2, &Font_AditChicago, (SVGUint8*)"Bosch Jeva Sans", 0, SVG_TOP );
//    printf("Selected font1: %s\n", Font_AditChicago.fontName);
    svgEnableShapingEngine(p_font_ctx2, SVG_SCRIPT_ARABIC ,SVG_RENDER_DIRECTION_RTL);
    svgLoadFont( p_font_ctx2, &Font_AditChicago, SVG_ENCODING_UTF8, 40);
    svgSetBodyColor(p_font_ctx2, 0xff0000ff);
    glClearColor(0.0f, 1.0f, 1.0f, 1.0f);				
    glClear(GL_COLOR_BUFFER_BIT);
    svgSetContourWidth(p_font_ctx2,1);
    svgDrawFontString(p_font_ctx2, 0.0 ,300.0 , &arabic_string[0], strlen((char*)arabic_string));
    eglSwapBuffers(eglDisplay, eglWindowSurface);
    sleep(10);
*/
    
    svgDestroyFontContext(p_font_ctx);
    svgCloseFonts();   

    eglClose();
#endif    
}


void draw_font_test (void)
{
    SVGUint8    *strings[] = {(SVGUint8*)(void*)"very first ", (SVGUint8*)(void*)"teststring ", (SVGUint8*)(void*)"for ", (SVGUint8*)(void*)"extended ", (SVGUint8*)(void*)"fontbitmap ", (SVGUint8*)(void*)"test "};
    SVGUint32   strings_len[6] = {13, 13, 13, 13, 13, 13};   
	SVGUint32	num_strings	  = 6;
    SVGFloat    x[]           = {100.0f, 100.0f, 100.0f, 100.0f, 100.0f, 100.0f};
    SVGFloat    y[]           = {450.0f, 420.0f, 390.0f, 360.0f, 330.0f, 300.0f};
    SVGUint32   loop       = 0;
    SVGFloat    tmp_angle  = 0.0;
    SVGBBox     a_box      = {0, 0, 0, 0};
    SVGBBox     a_box2     = {0, 0, 0, 0};
    SVGBBox     a_box3     = {0, 0, 0, 0};
    SVGUint32   maxchars   = 0;
    SVGUint32   maxchars2  = 0;
    SVGUint32   maxchars3  = 0;
    SVGFontInfo svgInfo    = {0, 0, 0, 0, SVG_ENCODING_MIN, 0, NULL, 0};
    SVGFloat    vertex[10] = {0};
    SVGFloat    texCoords[8] = {0.0f};
    SVGUint32   error;
     SVGBBox     **filledGlyphbox 		= (SVGBBox**) malloc( sizeof(SVGBBox*) * num_strings);
    SVGBBox     **filledTexturebox 		= (SVGBBox**) malloc( sizeof(SVGBBox*) * num_strings);
	SVGBBox     **strokedGlyphbox 		= (SVGBBox**) malloc( sizeof(SVGBBox*) * num_strings);
    SVGBBox     **strokedTexturebox 	= (SVGBBox**) malloc( sizeof(SVGBBox*) * num_strings);
	SVGUint32   filledTexture     = 0;
    SVGUint32   strokedTexture    = 0;
    SVGUint32   glyphcount[6] = {0};
    SVGUint32   texts      = 0;
    SVGUint32   loop_i     = 0;
    SVGUint32   delta_col  = 0;
    SVGUint32   col        = 0;
    SVGFloat    colors[4]  = {0.0f};
    SVGUint32   index_i      = 0;
    //SVGFloat    yy = 0;
    //SVGUint32   fs = 0;
#ifdef HAS_OPENVG
    VGImage     image      = 0;
#endif    

    svgDrawFontString(p_font_ctx2, 0.0 ,0.0 , (SVGUint8*)"0123456789", 10);
/*    svgSetFontRotate( p_font_ctx, 0.0);
    svgEnableFontSettings(p_font_ctx, SVG_FONT_KERNING);
    svgEnableFontSettings(p_font_ctx, SVG_FONT_HINTING);
    for(fs = 40; fs > 0 ; fs-=2)
	{
		svgSetFontSize(p_font_ctx, fs);
		svgDrawFontString(p_font_ctx, 0.0 ,yy , (SVGUint8*)"SD08G", 5);
        yy += (SVGFloat)fs;
	}*/

    for(loop = 0; loop < num_strings ; loop++)
	{
    	if((NULL != filledGlyphbox) && (NULL != filledTexturebox) && (NULL != strokedGlyphbox) && (NULL != strokedTexturebox)){
		filledGlyphbox[loop] = (SVGBBox*) malloc( sizeof(SVGBBox) * strings_len[loop]);
    	filledTexturebox[loop] = (SVGBBox*) malloc( sizeof(SVGBBox) * strings_len[loop]);
    	strokedGlyphbox[loop] = (SVGBBox*) malloc( sizeof(SVGBBox) * strings_len[loop]);
    	strokedTexturebox[loop] = (SVGBBox*) malloc( sizeof(SVGBBox) * strings_len[loop]);
    	}
	}
    /* get bounding box here for test purposes ->bounding box is equal to displayed string*/
    svgSetFontDrawingOrigin(p_font_ctx, SVG_ORIGIN_BASELINE);
    svgEnableFontSettings(p_font_ctx, SVG_FONT_2COLOR);
    /* set stroke color so 2 color strings and circle text outline is black */
    svgSetContourColor(p_font_ctx, 0x000000FF);

    svgEnableFontSettings(p_font_ctx, SVG_FONT_NO_ENCLOSING_SPACES);
    svgEnableFontSettings(p_font_ctx, SVG_FONT_SCALE);
    svgGetFontBBox(p_font_ctx, &a_string[0], sizeof(a_string), &a_box2);
    maxchars2 = svgGetFontMaxChars(p_font_ctx, a_box2.width, a_box2.height, &a_string[0], sizeof(a_string));
    svgDisableFontSettings(p_font_ctx, SVG_FONT_NO_ENCLOSING_SPACES);
    svgSetFontScale(p_font_ctx, 0.001f, 1.0f);
    svgSetFontScale(p_font_ctx, (sin(( 3.14159 * 2 * a_angle ) / 360)*sin(( 3.14159 * 2 * a_angle ) / 360)), 1.0);
    svgGetFontBBox(p_font_ctx, &a_string[0], sizeof(a_string), &a_box);
    maxchars = svgGetFontMaxChars(p_font_ctx, a_box.width, a_box.height, &a_string[0], sizeof(a_string));

    for(loop = 0;loop < sizeof(a_string2); loop++)
    {
        tmp_angle = 360.0 - (a_angle - ((SVGFloat)(loop) * (360.0 / (sizeof(a_string2)+1))));
        angles[loop] = tmp_angle + 270.0;
        while(tmp_angle < 0.0)
        {
          tmp_angle += 360.0;
        }
        while(tmp_angle >= 360.0)
        {
          tmp_angle -= 360.0;
        }
        pos[(2 * loop)] = 600 - (100 * cos(( 3.14159 * 2 * tmp_angle ) / 360));
        pos[(2 * loop) + 1] = 300 + (100 * sin(( 3.14159 * 2 * tmp_angle) / 360));
    }
    
    /* red 2-color string */
    svgDrawFontWithCursorInfo(p_font_ctx, 200.0 ,200.0 , &a_string[0], sizeof(a_string), (SVGPoint*)&pos2[0]);
    /* draw in magenta the cursor positions and the bounding box*/

    if(0 != use_opengl)
    {
#ifdef HAS_OPENGLES
        colors[0] = 1.0;
        colors[1] = 0.0;
        colors[2] = 1.0;
        colors[3] = 1.0;
        error =glGetError();
        if( error != GL_NO_ERROR)
        {
                printf("GL ERROR = %d\n",error);
        }
        glUseProgram(hProgramHandle[0]);

        error =glGetError();
        if( error != GL_NO_ERROR)
        {
                printf("GL ERROR = %d\n",error);
        }
        glUniform4fv(colorlocation, 1, &colors[0]);

        error =glGetError();
        if( error != GL_NO_ERROR)
        {
                printf("GL ERROR = %d\n",error);
        }

        for(loop = 0;loop < sizeof(a_string) - 1; loop++)
        {
            vertex[0] = pos2[(2 * loop)];
            vertex[1] = pos2[(2 * loop) + 1];
            vertex[2] = pos2[(2 * loop)] + ( 40 * sin(( 3.14159 * 2 * a_angle ) / 360));
            vertex[3] = pos2[(2 * loop) + 1] + (40 * cos(( 3.14159 * 2 * a_angle ) / 360));    
            glUniformMatrix4fv(mvplocation,  1, GL_FALSE, &projection[0][0]);
            error =glGetError();
            if( error != GL_NO_ERROR)
            {
                    printf("GL ERROR = %d\n",error);
            }
            glVertexAttribPointer(0, 2, GL_FLOAT, 0, 0, vertex);
            error =glGetError();
            if( error != GL_NO_ERROR)
            {
                    printf("GL ERROR = %d\n",error);
            }
            glDrawArrays(GL_LINES, 0, 2);
        }

        vertex[0] = (200.0 + a_box.left);
        vertex[1] = (200.0 + a_box.top);
        vertex[2] = vertex[0] + a_box.width;
        vertex[3] = vertex[1];
        vertex[4] = vertex[2];
        vertex[5] = vertex[3] + a_box.height;
        vertex[6] = vertex[0];
        vertex[7] = vertex[5];
        glUniformMatrix4fv(mvplocation,  1, GL_FALSE, &projection[0][0]);
        glVertexAttribPointer(0, 2, GL_FLOAT, 0, 0, vertex);
        
        glDrawArrays(GL_LINE_LOOP/*GL_TRIANGLE_FAN*/, 0, 4);
        svgSetBodyColor(p_font_ctx, 0xffffffff);        
#endif
    }
    else
    {
#ifdef HAS_OPENVG
    	vgSetColor (testpaint,  0xFF00FFFF);
        for(loop = 0;loop < sizeof(a_string) - 1; loop++)
        {
            vgClearPath(a_path,VG_PATH_CAPABILITY_ALL);
            vguLine (a_path, pos2[(2 * loop)], pos2[(2 * loop) + 1], pos2[(2 * loop)] + ( 40 * sin(( 3.14159 * 2 * a_angle ) / 360)), pos2[(2 * loop) + 1] + (40 * cos(( 3.14159 * 2 * a_angle ) / 360)));    
            vgDrawPath (a_path, VG_STROKE_PATH);
        }
    
        /* draw bounding box of red 2-color string */
        vgClearPath(a_path,VG_PATH_CAPABILITY_ALL);
        vguRect (a_path, 200.0 + a_box.left, 200.0 + a_box.top, a_box.width, a_box.height);    
        vgDrawPath (a_path, VG_STROKE_PATH);
        /* set fill color to white so filled part of circle text is white*/
    	vgSetColor (testpaint,  0xFFFFFFFF);
#endif
    }

    svgSetFontScale(p_font_ctx, 1.0, 0.5);
    /* the "circle" text */
    svgDrawFontStringExt(p_font_ctx, (SVGPoint*)&pos[0], &angles[0], &a_string2[0], sizeof(a_string2));
    svgDisableFontSettings(p_font_ctx, SVG_FONT_SCALE);
    svgDisableFontSettings(p_font_ctx, SVG_FONT_2COLOR);

    /* green outline string */
    svgEnableFontSettings(p_font_ctx, SVG_FONT_OUTLINE);
    svgEnableFontSettings(p_font_ctx, SVG_FONT_NO_ENCLOSING_SPACES);
    svgEnableFontSettings(p_font_ctx, SVG_FONT_OPACITY);
    svgSetFontOpacity(p_font_ctx, (cos(( 3.14159 * 2 * a_angle ) / 360) * cos(( 3.14159 * 2 * a_angle ) / 360)));
    svgSetFontDrawingOrigin(p_font_ctx, SVG_ORIGIN_TOP);
    svgSetContourColor(p_font_ctx, 0x00FF00FF);
    svgDrawFontString(p_font_ctx, 200.0 ,200.0 , &a_string1[0], sizeof(a_string1));
    svgDisableFontSettings(p_font_ctx, SVG_FONT_OPACITY);
    svgGetFontBBox(p_font_ctx, &a_string1[0], sizeof(a_string1), &a_box3);
    /* draw bounding box of green outline string  in green */
    if(0 != use_opengl)
    {
#ifdef HAS_OPENGLES
       colors[0] = 0.0;
       colors[1] = 1.0;
       colors[2] = 0.0;
       colors[3] = 1.0;
       glUseProgram(hProgramHandle[0]);
       glUniform4fv(colorlocation, 1, &colors[0]);
        vertex[0] = (200.0 + a_box3.left);
        vertex[1] = (200.0 + a_box3.top);
        vertex[2] = vertex[0] + a_box3.width ;
        vertex[3] = vertex[1];
        vertex[4] = vertex[2];
        vertex[5] = vertex[3] + a_box3.height;
        vertex[6] = vertex[0];
        vertex[7] = vertex[5];
        glUniformMatrix4fv(mvplocation,  1, GL_FALSE, &projection[0][0]);
        glVertexAttribPointer(0, 2, GL_FLOAT, 0, 0, vertex);
        glDrawArrays(GL_LINE_LOOP, 0, 4);
        svgGetFontInformation(p_font_ctx, &svgInfo);
        svgSetBodyColor(p_font_ctx, 0xff0000ff);        
#endif
    }
    else
    {
#ifdef HAS_OPENVG
    	vgSetColor (testpaint,  0x00FF00FF);
        vgClearPath(a_path,VG_PATH_CAPABILITY_ALL);
        svgGetFontInformation(p_font_ctx, &svgInfo);
        vguRect (a_path, 200.0 + a_box3.left, 200.0 + a_box3.top, a_box3.width, a_box3.height);    
        vgDrawPath (a_path, VG_STROKE_PATH);
        /* set fill color to red, so 2-color string filled area is red */
    	vgSetColor (testpaint,  0xFF0000FF);
#endif    	
    }

    /* white vertical outline string */
    svgEnableFontSettings(p_font_ctx, SVG_FONT_VERTICAL);
    svgSetContourColor(p_font_ctx, 0xFFFFFFFF);
    svgDrawFontString(p_font_ctx, 200.0 ,200.0 , &a_string3[0], sizeof(a_string3));
    svgDisableFontSettings(p_font_ctx, SVG_FONT_OPACITY);
    svgGetFontBBox(p_font_ctx, &a_string3[0], sizeof(a_string3), &a_box3);
    svgDisableFontSettings(p_font_ctx, SVG_FONT_NO_ENCLOSING_SPACES);
    svgDisableFontSettings(p_font_ctx, SVG_FONT_VERTICAL);
    /* draw bounding box of green outline string  in green */
    if(0 != use_opengl)
    {
#ifdef HAS_OPENGLES
       colors[0] = 1.0;
       colors[1] = 1.0;
       colors[2] = 1.0;
       colors[3] = 1.0;
       glUseProgram(hProgramHandle[0]);
       glUniform4fv(colorlocation, 1, &colors[0]);
        vertex[0] = (200.0 + a_box3.left);
        vertex[1] = (200.0 + a_box3.top);
        vertex[2] = vertex[0] + a_box3.width ;
        vertex[3] = vertex[1];
        vertex[4] = vertex[2];
        vertex[5] = vertex[3] + a_box3.height;
        vertex[6] = vertex[0];
        vertex[7] = vertex[5];
        glUniformMatrix4fv(mvplocation,  1, GL_FALSE, &projection[0][0]);
        glVertexAttribPointer(0, 2, GL_FLOAT, 0, 0, vertex);
        glDrawArrays(GL_LINE_LOOP, 0, 4);
        svgGetFontInformation(p_font_ctx, &svgInfo);
        svgSetBodyColor(p_font_ctx, 0xff0000ff);        
#endif
    }
    else
    {
#ifdef HAS_OPENVG
    	vgSetColor (testpaint,  0xFFFFFFFF);
        vgClearPath(a_path,VG_PATH_CAPABILITY_ALL);
        vguRect (a_path, 200.0 + a_box3.left, 200.0 + a_box3.top, a_box3.width, a_box3.height);    
        vgDrawPath (a_path, VG_STROKE_PATH);
        /* set fill color to red, so 2-color string filled area is red */
    	vgSetColor (testpaint,  0xFF0000FF);
#endif    	
    }

    maxchars3 = svgGetFontMaxChars(p_font_ctx, a_box3.width, a_box3.height, &a_string1[0], sizeof(a_string1));
#if 0
    /*TODO: check if maxchars values are ok*/
    printf("TEST svg_font_demo: maxchars = %d\n",maxchars);
    printf("TEST svg_font_demo: maxchars2 = %d\n",maxchars2);
    printf("TEST svg_font_demo: maxchars3 = %d\n",maxchars3);
#else
    maxchars = maxchars;
    maxchars2 = maxchars2;
    maxchars3 = maxchars3;
#endif
#if 0
    svgSetFontScale(p_font_ctx, 1.0, 1.0);
    svgSetFontRotate(p_font_ctx, (0));
    svgDisableFontSettings(p_font_ctx, SVG_FONT_NO_ENCLOSING_SPACES);
    svgGetFontBBox(p_font_ctx, "93.0", 4, &a_box);
    printf("outl 93.0: width : %f\n",a_box.width);
    svgGetFontBBox(p_font_ctx, "93.1", 4, &a_box);
    printf("outl 93.1: width : %f\n",a_box.width);
    svgGetFontBBox(p_font_ctx, "93.2", 4, &a_box);
    printf("outl 93.2: width : %f\n",a_box.width);

    svgEnableFontSettings(p_font_ctx, SVG_FONT_NO_ENCLOSING_SPACES);
    svgGetFontBBox(p_font_ctx, "93.0", 4, &a_box);
    printf("outl no 93.0: width : %f\n",a_box.width);
    svgGetFontBBox(p_font_ctx, "93.1", 4, &a_box);
    printf("outl no 93.1: width : %f\n",a_box.width);
    svgGetFontBBox(p_font_ctx, "93.2", 4, &a_box);
    printf("outl no 93.2: width : %f\n",a_box.width);

    svgDisableFontSettings(p_font_ctx, SVG_FONT_NO_ENCLOSING_SPACES);
    svgDisableFontSettings(p_font_ctx, SVG_FONT_OUTLINE);
    svgGetFontBBox(p_font_ctx, "93.0", 4, &a_box);
    printf("93.0: width : %f\n",a_box.width);
    svgGetFontBBox(p_font_ctx, "93.1", 4, &a_box);
    printf("93.1: width : %f\n",a_box.width);
    svgGetFontBBox(p_font_ctx, "93.2", 4, &a_box);
    printf("93.2: width : %f\n",a_box.width);

    svgEnableFontSettings(p_font_ctx, SVG_FONT_NO_ENCLOSING_SPACES);
    svgGetFontBBox(p_font_ctx, "93.0", 4, &a_box);
    printf("no 93.0: width : %f\n",a_box.width);
    svgGetFontBBox(p_font_ctx, "93.1", 4, &a_box);
    printf("no 93.1: width : %f\n",a_box.width);
    svgGetFontBBox(p_font_ctx, "93.2", 4, &a_box);
    printf("no 93.2: width : %f\n",a_box.width);
    svgSetFontRotate(p_font_ctx, (a_angle));
    svgDisableFontSettings(p_font_ctx, SVG_FONT_NO_ENCLOSING_SPACES);
    svgSetFontScale(p_font_ctx, 1.0, 0.5);
    svgEnableFontSettings(p_font_ctx, SVG_FONT_OUTLINE);
#endif

#if 1
    texts = svgGetFontBitmapExt(p_font_ctx,
                                &strings[0],
                                &strings_len[0],
                                num_strings,
                                &filledTexture,
								&strokedTexture,
								filledTexturebox,
								strokedTexturebox,
								filledGlyphbox,
								strokedGlyphbox,
                                &glyphcount[0]);

    loop_i = 0;
    for(loop = 0; loop < texts; loop++)
    {
        loop_i += glyphcount[loop];
    }
    loop_i--;
//    -> different color for each glyph;
    delta_col = 0xFFFFFF / loop_i;
    vertex[0] = 0.0f;
    vertex[1] = 0.0f;
    vertex[3] = 0.0f;
    vertex[6] = 0.0f;
 
    if(0 != use_opengl)
    {
#ifdef HAS_OPENGLES2
        glUseProgram(hProgramHandle[1]);
        glUniform1f(opacitylocation, 1.0f);
        glUniform1i(noaalocation, 0);
#endif    
    }
    for(loop = 0; loop < texts; loop++)
    {
        if(0 != use_opengl)
        {
#ifdef HAS_OPENGLES2
            demo_matrixIdentity(modelview);
            demo_matrixTranslate(modelview, x[loop],y[loop]);
#endif
        }
        else
        {
#ifdef HAS_OPENVG
        vgLoadIdentity();
        vgTranslate (x[loop], y[loop]);
#endif        
        }
		for(loop_i = 0; loop_i < glyphcount[loop]; loop_i++)
        {
            colors[0] = (SVGFloat)((col >> 16) & 0xff) / 255.0f;
            colors[1] = (SVGFloat)((col >> 8) & 0xff) / 255.0f;
            colors[2] = (SVGFloat)(col  & 0xff) / 255.0f;
            colors[3] = 1.0f;
            if(0 != use_opengl)
            {
#ifdef HAS_OPENGLES2
                glUniform4fv(colorlocation1, 1, &colors[0]);
            	if (NULL != strokedGlyphbox){
                demo_matrixTranslate(modelview, strokedGlyphbox[loop][loop_i].left, strokedGlyphbox[loop][loop_i].top);
            	}
            	demo_matrixMutiply(mvp, modelview, projection);
                glUniformMatrix4fv(mvplocation1, 1, GL_FALSE, &mvp[0][0]);
#endif      
            }

            if(((NULL != strokedGlyphbox) && (NULL != strokedTexturebox)) && 0 != (strokedTexturebox[loop][loop_i].width * strokedTexturebox[loop][loop_i].height))
            {   
                vertex[2] = (strokedGlyphbox[loop][loop_i].width);
                vertex[4] = vertex[2];
                vertex[5] = (strokedGlyphbox[loop][loop_i].height);
                vertex[7] = vertex[5];
            
                /* source vertex */
                texCoords[0] = strokedTexturebox[loop][loop_i].left;
                texCoords[1] = strokedTexturebox[loop][loop_i].top;
                texCoords[2] = texCoords[0] + strokedTexturebox[loop][loop_i].width;
                texCoords[3] = texCoords[1];
                texCoords[4] = texCoords[2];
                texCoords[5] = texCoords[3] + strokedTexturebox[loop][loop_i].height;
                texCoords[6] = texCoords[0];
                texCoords[7] = texCoords[5];
    
                if(0 != use_opengl)
                {
#ifdef HAS_OPENGLES2
                    glVertexAttribPointer(texturelocation, 2, GL_FLOAT, 0, 0, texCoords);
                    glVertexAttribPointer(vertexlocation, 2, GL_FLOAT, 0, 0, vertex);
                    glDrawArrays(GL_TRIANGLE_FAN, 0, 4);
#endif        
                }
                else
                {
#ifdef HAS_OPENVG
                    vgSetParameterfv(testpaint, VG_PAINT_COLOR, 4, colors);
                    if ((NULL != strokedGlyphbox) && (NULL != strokedTexturebox)){
                    vgTranslate(strokedGlyphbox[loop][loop_i].left, strokedGlyphbox[loop][loop_i].top);
                    image = vgChildImage(strokedTexture,
                                      (SVGInt32)strokedTexturebox[loop][loop_i].left,
                                      (SVGInt32)strokedTexturebox[loop][loop_i].top,
                                      (SVGInt32)strokedTexturebox[loop][loop_i].width,
                                      (SVGInt32)strokedTexturebox[loop][loop_i].height);
                    }
                    vgDrawImage(image);
                    vgDestroyImage(image);
#endif    	
                }
            }
            col += delta_col;
            index_i++;
        }    
    }
	if(0 != use_opengl)
    {
#ifdef HAS_OPENGLES2
        glUseProgram(hProgramHandle[0]);
        colors[0] = 1.0;
        colors[1] = 1.0;
        colors[2] = 1.0;
        colors[3] = 1.0;
        glUniform4fv(colorlocation, 1, &colors[0]);
        svgSetBodyColor(p_font_ctx, 0xff0000ff);        
#endif        
    }
    else
    {
#ifdef HAS_OPENVG
        vgLoadIdentity();
    	vgSetColor (testpaint,  0xFF0000FF);
#endif    	
    }

#endif
	for(loop = 0; loop < num_strings ; loop++)
	{
		if((NULL != filledGlyphbox) && (NULL != filledTexturebox) && (NULL != strokedGlyphbox) && (NULL != strokedTexturebox)){
		free( filledGlyphbox[loop]);
		free( filledTexturebox[loop]);
		free( strokedGlyphbox[loop]);
		free( strokedTexturebox[loop]);
		}
	}
	free( filledGlyphbox);
	free( strokedGlyphbox );
	free( filledTexturebox);
	free( strokedTexturebox);
	
    svgDisableFontSettings(p_font_ctx, SVG_FONT_OUTLINE);
    a_angle += 10.0;
    if(a_angle >= 360.0)
    {
      a_angle = 0.0;
    }
    svgSetFontRotate(p_font_ctx, (a_angle));

}
